<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use Erlage\Photogram\Tools\CookieManager;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;

final class Session
{
    /**
     * @return Session 
     */
    final public static function instance()
    {
        static $instance = false;

        if (false === $instance)
        {
            $instance = new Session();
        }

        return $instance;
    }

    /**
     * @var bool
     */
    protected $isAuthenticated = false;

    /**
     * @var UserModel
     */
    protected $userModel;

    public function init(): void
    {
        $userId = CookieManager::get(UserTable::TABLE_NAME . '_' . UserTable::ID);

        if (null != $userId)
        {
            $userModel = UserModel::findFromId_noException($userId);

            if ($userModel -> isModel())
            {
                // this will also verify meta access token.
                $this -> setUserModel($userModel);

                if ($this -> isAuthenticated())
                {
                    UserModelHelper::updateLastActive($this -> getUserModel());
                }
            }
        }
    }

    public function isAuthenticated(): bool
    {
        return $this -> isAuthenticated;
    }

    public function getUserModel(): UserModel
    {
        return $this -> userModel;
    }

    public function loginAs(UserModel $userModel): void
    {
        $this -> userModel = $userModel;

        $this -> isAuthenticated = true;
    }

    public function setUserModel(UserModel $userModel): void
    {
        $this -> userModel = $userModel;

        $this -> verifyMetaAccessToken();
    }

    public function logout()
    {
        $this -> isAuthenticated = false;

        $this -> removeCookies();
    }

    public function updateSession()
    {
        if ($this -> isAuthenticated())
        {
            $this -> addCookies();
        }
    }

    private function verifyMetaAccessToken(): void
    {
        if ($this -> getUserModel() -> isModel())
        {
            $adminMetaAccess = CookieManager::get(UserTable::TABLE_NAME . '_' . UserTable::META_ACCESS);

            if ($adminMetaAccess == $this -> getUserModel() -> getMetaAccess())
            {
                $this -> isAuthenticated = true;
            }
        }
        else
        {
            $this -> logout();
        }
    }

    private function addCookies()
    {
        CookieManager::set(
            UserTable::TABLE_NAME . '_' . UserTable::ID,
            $this -> getUserModel() -> getId()
        );

        CookieManager::set(
            UserTable::TABLE_NAME . '_' . UserTable::META_ACCESS,
            $this -> getUserModel() -> getMetaAccess()
        );
    }

    private function removeCookies()
    {
        CookieManager::unset(UserTable::TABLE_NAME . '_' . UserTable::ID);

        CookieManager::unset(UserTable::TABLE_NAME . '_' . UserTable::META_ACCESS);
    }
}
